using System;
using System.Collections.Generic;
using System.Globalization;

/// <summary>
/// Ponco - Approximate Knapsack Problem
/// 
/// For n = 40: Meet-in-the-Middle (exact, O(2^(n/2)))
/// For n > 40: FPTAS with smart parameter choice
/// 
/// Key insight for FPTAS:
/// - We scale values by K = e * maxV / n
/// - This gives at most n/e distinct scaled values per item
/// - Total reachable states = n * (n/e) = n²/e
/// 
/// For worst case n=300, e=0.01: n²/e = 9M - but we use array DP
/// which is fast in practice.
/// </summary>
class Ponco
{
    static int n;
    static long W;
    static double epsilon;
    static long[] weight, value;
    
    static (long totalValue, List<int> items) MeetInTheMiddle()
    {
        int half = n / 2;
        int left = half;
        int right = n - half;
        
        var leftList = new List<(long w, long v, int mask)>();
        for (int mask = 0; mask < (1 << left); mask++)
        {
            long w = 0, v = 0;
            bool ok = true;
            for (int i = 0; i < left && ok; i++)
            {
                if ((mask & (1 << i)) != 0)
                {
                    w += weight[i];
                    v += value[i];
                    if (w > W) ok = false;
                }
            }
            if (ok && w <= W)
                leftList.Add((w, v, mask));
        }
        
        leftList.Sort((a, b) => a.w.CompareTo(b.w));
        
        var hull = new List<(long w, long v, int mask)>();
        long bestV = -1;
        foreach (var s in leftList)
        {
            if (s.v > bestV)
            {
                bestV = s.v;
                hull.Add(s);
            }
        }
        
        long maxValue = 0;
        int leftRes = 0, rightRes = 0;
        
        for (int mask = 0; mask < (1 << right); mask++)
        {
            long w = 0, v = 0;
            bool ok = true;
            for (int i = 0; i < right && ok; i++)
            {
                if ((mask & (1 << i)) != 0)
                {
                    w += weight[half + i];
                    v += value[half + i];
                    if (w > W) ok = false;
                }
            }
            if (!ok || w > W) continue;
            
            long remW = W - w;
            int lo = 0, hi = hull.Count - 1, best = -1;
            while (lo <= hi)
            {
                int mid = (lo + hi) / 2;
                if (hull[mid].w <= remW) { best = mid; lo = mid + 1; }
                else hi = mid - 1;
            }
            
            long totalV = v + (best >= 0 ? hull[best].v : 0);
            if (totalV > maxValue)
            {
                maxValue = totalV;
                leftRes = best >= 0 ? hull[best].mask : 0;
                rightRes = mask;
            }
        }
        
        List<int> items = new List<int>();
        for (int i = 0; i < left; i++)
            if ((leftRes & (1 << i)) != 0) items.Add(i + 1);
        for (int i = 0; i < right; i++)
            if ((rightRes & (1 << i)) != 0) items.Add(half + i + 1);
        
        return (maxValue, items);
    }
    
    /// <summary>
    /// FPTAS with array-based DP
    /// Scale factor chosen to balance accuracy and speed
    /// </summary>
    static (long totalValue, List<int> items) FPTASArrayDP()
    {
        // Find max value
        long maxV = 0;
        for (int i = 0; i < n; i++)
            if (value[i] > maxV) maxV = value[i];
        
        if (maxV == 0) return (0, new List<int>());
        
        // For FPTAS: K = e * maxV / n
        // Scaled value = floor(v / K) = floor(v * n / (e * maxV))
        // Max scaled value per item = n/e
        // But we want to limit total DP size
        
        // Calculate scaled values
        double scale = (double)n / (epsilon * maxV);
        int[] scaledV = new int[n];
        int maxScaled = 0;
        
        for (int i = 0; i < n; i++)
        {
            scaledV[i] = (int)(value[i] * scale);
            if (scaledV[i] < 1 && value[i] > 0) scaledV[i] = 1;
            maxScaled += scaledV[i];
        }
        
        // Limit DP size
        int dpSize = Math.Min(maxScaled + 1, 5000001);
        
        // dp[v] = min weight to achieve value v
        long[] dp = new long[dpSize];
        short[] lastItem = new short[dpSize]; // Use short to save memory
        int[] prevVal = new int[dpSize];
        
        for (int i = 0; i < dpSize; i++)
        {
            dp[i] = long.MaxValue;
            lastItem[i] = -1;
            prevVal[i] = -1;
        }
        dp[0] = 0;
        
        int maxReached = 0;
        
        for (int i = 0; i < n; i++)
        {
            int sv = scaledV[i];
            if (sv == 0) continue;
            
            long wi = weight[i];
            
            // Process in reverse to avoid using same item twice
            int limit = Math.Min(maxReached + sv, dpSize - 1);
            for (int v = limit; v >= sv; v--)
            {
                int prev = v - sv;
                if (dp[prev] != long.MaxValue)
                {
                    long newW = dp[prev] + wi;
                    if (newW <= W && newW < dp[v])
                    {
                        dp[v] = newW;
                        lastItem[v] = (short)i;
                        prevVal[v] = prev;
                        if (v > maxReached) maxReached = v;
                    }
                }
            }
        }
        
        // Find best achievable value
        int bestV = 0;
        for (int v = maxReached; v >= 0; v--)
        {
            if (dp[v] != long.MaxValue) { bestV = v; break; }
        }
        
        // Reconstruct
        List<int> items = new List<int>();
        int c = bestV;
        while (c > 0 && lastItem[c] >= 0)
        {
            items.Add(lastItem[c] + 1);
            c = prevVal[c];
        }
        
        // Calculate actual value
        long total = 0;
        foreach (int idx in items) total += value[idx - 1];
        
        return (total, items);
    }
    
    static (long totalValue, List<int> items) Greedy()
    {
        // Sort indices by value/weight ratio
        var indices = new int[n];
        var ratios = new double[n];
        for (int i = 0; i < n; i++)
        {
            indices[i] = i;
            ratios[i] = (double)value[i] / weight[i];
        }
        Array.Sort(ratios, indices);
        Array.Reverse(indices);
        
        long totalW = 0, totalV = 0;
        List<int> items = new List<int>();
        
        for (int j = 0; j < n; j++)
        {
            int i = indices[j];
            if (totalW + weight[i] <= W)
            {
                totalW += weight[i];
                totalV += value[i];
                items.Add(i + 1);
            }
        }
        
        return (totalV, items);
    }
    
    static (long totalValue, List<int> items) SingleBest()
    {
        long bestV = 0;
        int bestIdx = -1;
        for (int i = 0; i < n; i++)
        {
            if (weight[i] <= W && value[i] > bestV)
            {
                bestV = value[i];
                bestIdx = i;
            }
        }
        if (bestIdx == -1) return (0, new List<int>());
        return (bestV, new List<int> { bestIdx + 1 });
    }
    
    static void Main()
    {
        string[] first = Console.ReadLine().Split(new char[] { ' ', '\t' }, StringSplitOptions.RemoveEmptyEntries);
        n = int.Parse(first[0]);
        W = long.Parse(first[1]);
        epsilon = double.Parse(first[2], CultureInfo.InvariantCulture);
        
        weight = new long[n];
        value = new long[n];
        
        for (int i = 0; i < n; i++)
        {
            string[] p = Console.ReadLine().Split(new char[] { ' ', '\t' }, StringSplitOptions.RemoveEmptyEntries);
            weight[i] = long.Parse(p[0]);
            value[i] = long.Parse(p[1]);
        }
        
        // Collect results from different strategies
        long bestVal = 0;
        List<int> bestItems = new List<int>();
        
        // Strategy 1: Meet in the Middle for small n
        if (n <= 40)
        {
            var (v, items) = MeetInTheMiddle();
            if (v > bestVal) { bestVal = v; bestItems = items; }
        }
        
        // Strategy 2: FPTAS
        {
            var (v, items) = FPTASArrayDP();
            if (v > bestVal) { bestVal = v; bestItems = items; }
        }
        
        // Strategy 3: Greedy
        {
            var (v, items) = Greedy();
            if (v > bestVal) { bestVal = v; bestItems = items; }
        }
        
        // Strategy 4: Single best item
        {
            var (v, items) = SingleBest();
            if (v > bestVal) { bestVal = v; bestItems = items; }
        }
        
        // Output
        Console.WriteLine(bestItems.Count);
        if (bestItems.Count > 0)
            Console.WriteLine(string.Join(" ", bestItems));
        else
            Console.WriteLine();
    }
}